<?php
/**
 * Name: Advanced content Filter
 * Description: Expression-based content filter
 * Version: 1.0
 * Author: Hypolite Petovan <https://friendica.mrpetovan.com/profile/hypolite>
 * Maintainer: Hypolite Petovan <https://friendica.mrpetovan.com/profile/hypolite>
 *
 * Copyright (c) 2018 Hypolite Petovan
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *    * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *    * Redistributions in binary form must reproduce the above
 *    * copyright notice, this list of conditions and the following disclaimer in
 *      the documentation and/or other materials provided with the distribution.
 *    * Neither the name of Friendica nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL FRIENDICA BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

use Friendica\BaseModule;
use Friendica\Content\Text\Markdown;
use Friendica\Core\Hook;
use Friendica\Core\Renderer;
use Friendica\Database\DBA;
use Friendica\Database\DBStructure;
use Friendica\DI;
use Friendica\Model\Post;
use Friendica\Model\Tag;
use Friendica\Model\User;
use Friendica\Module\Security\Login;
use Friendica\Network\HTTPException;
use Friendica\Util\DateTimeFormat;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Symfony\Component\ExpressionLanguage;

require_once __DIR__ . DIRECTORY_SEPARATOR . 'vendor' . DIRECTORY_SEPARATOR . 'autoload.php';

function advancedcontentfilter_install()
{
	Hook::register('dbstructure_definition'     , __FILE__, 'advancedcontentfilter_dbstructure_definition');
	Hook::register('prepare_body_content_filter', __FILE__, 'advancedcontentfilter_prepare_body_content_filter');
	Hook::register('addon_settings'             , __FILE__, 'advancedcontentfilter_addon_settings');

	Hook::add('dbstructure_definition'          , __FILE__, 'advancedcontentfilter_dbstructure_definition');
	DBStructure::performUpdate();

	DI::logger()->notice('installed advancedcontentfilter');
}

/*
 * Hooks
 */

function advancedcontentfilter_dbstructure_definition(&$database)
{
	$database['advancedcontentfilter_rules'] = [
		'comment' => 'Advancedcontentfilter addon rules',
		'fields' => [
			'id'         => ['type' => 'int unsigned', 'not null' => '1', 'extra' => 'auto_increment', 'primary' => '1', 'comment' => 'Auto incremented rule id'],
			'uid'        => ['type' => 'int unsigned', 'not null' => '1', 'comment' => 'Owner user id'],
			'name'       => ['type' => 'varchar(255)', 'not null' => '1', 'comment' => 'Rule name'],
			'expression' => ['type' => 'mediumtext'  , 'not null' => '1', 'comment' => 'Expression text'],
			'serialized' => ['type' => 'mediumtext'  , 'not null' => '1', 'comment' => 'Serialized parsed expression'],
			'active'     => ['type' => 'boolean'     , 'not null' => '1', 'default' => '1', 'comment' => 'Whether the rule is active or not'],
			'created'    => ['type' => 'datetime'    , 'not null' => '1', 'default' => DBA::NULL_DATETIME, 'comment' => 'Creation date'],
		],
		'indexes' => [
			'PRIMARY' => ['id'],
			'uid_active' => ['uid', 'active'],
		]
	];
}

/**
 * @param array $item Prepared by either Model\Item::prepareBody or advancedcontentfilter_prepare_item_row
 * @return array
 */
function advancedcontentfilter_get_filter_fields(array $item)
{
	$vars = [];

	// Convert the language JSON text into a filterable format
	if (!empty($item['language']) && ($languages = json_decode($item['language'], true))) {
		foreach ($languages as $key => $value) {
			$vars['language_' . strtolower($key)] = $value;
		}
	}

	foreach ($item as $key => $value) {
		$vars[str_replace('-', '_', $key)] = $value;
	}

	ksort($vars);

	return $vars;
}

function advancedcontentfilter_prepare_body_content_filter(&$hook_data)
{
	static $expressionLanguage;

	if (is_null($expressionLanguage)) {
		$expressionLanguage = new ExpressionLanguage\ExpressionLanguage();
	}

	$uid = $hook_data['uid'] ?? DI::userSession()->getLocalUserId();
	if (!$uid) {
		return;
	}

	$vars = advancedcontentfilter_get_filter_fields($hook_data['item']);

	$rules = DI::cache()->get('rules_' . $uid);
	if (!isset($rules)) {
		$rules = DBA::toArray(DBA::select(
			'advancedcontentfilter_rules',
			['name', 'expression', 'serialized'],
			['uid' => $uid, 'active' => true]
		));

		DI::cache()->set('rules_' . $uid, $rules);
	}

	if ($rules) {
		foreach($rules as $rule) {
			try {
				$serializedParsedExpression = new ExpressionLanguage\SerializedParsedExpression(
					$rule['expression'],
					$rule['serialized']
				);

				// The error suppression operator is used because of potentially broken user-supplied regular expressions
				$found = (bool) @$expressionLanguage->evaluate($serializedParsedExpression, $vars);
			} catch (Exception $e) {
				$found = false;
			}

			if ($found) {
				$hook_data['filter_reasons'][] = DI::l10n()->t('Filtered by rule: %s', $rule['name']);
				break;
			}
		}
	}
}


function advancedcontentfilter_addon_settings(array &$data)
{
	if (!DI::userSession()->getLocalUserId()) {
		return;
	}

	$data = [
		'addon' => 'advancedcontentfilter',
		'title' => DI::l10n()->t('Advanced Content Filter'),
		'href'  => 'advancedcontentfilter',
	];
}

/*
 * Module
 */

/**
 * This is a statement rather than an actual function definition. The simple
 * existence of this method is checked to figure out if the addon offers a
 * module.
 */
function advancedcontentfilter_module() {}

function advancedcontentfilter_init()
{
	if (DI::args()->getArgc() > 1 && DI::args()->getArgv()[1] == 'api') {
		$slim = \Slim\Factory\AppFactory::create();

		/**
		 * The routing middleware should be added before the ErrorMiddleware
		 * Otherwise exceptions thrown from it will not be handled
		 */
		$slim->addRoutingMiddleware();

		$slim->addErrorMiddleware(true, true, true, DI::logger());

		// register routes
		$slim->group('/advancedcontentfilter/api', function (\Slim\Routing\RouteCollectorProxy $app) {
			$app->group('/rules', function (\Slim\Routing\RouteCollectorProxy $app) {
				$app->get('', 'advancedcontentfilter_get_rules');
				$app->post('', 'advancedcontentfilter_post_rules');

				$app->get('/{id}', 'advancedcontentfilter_get_rules_id');
				$app->put('/{id}', 'advancedcontentfilter_put_rules_id');
				$app->delete('/{id}', 'advancedcontentfilter_delete_rules_id');
			});

			$app->group('/variables', function (\Slim\Routing\RouteCollectorProxy $app) {
				$app->get('/{guid}', 'advancedcontentfilter_get_variables_guid');
			});
		});

		$slim->run();

		exit;
	}
}

function advancedcontentfilter_content()
{
	if (!DI::userSession()->getLocalUserId()) {
		return Login::form('/' . implode('/', DI::args()->getArgv()));
	}

	if (DI::args()->getArgc() > 1 && DI::args()->getArgv()[1] == 'help') {
		$user = User::getById(DI::userSession()->getLocalUserId());

		$lang = $user['language'];

		$default_dir = 'addon/advancedcontentfilter/doc/';
		$help_file = 'advancedcontentfilter.md';
		$help_path = $default_dir . $help_file;
		if (file_exists($default_dir . $lang . '/' . $help_file)) {
			$help_path = $default_dir . $lang . '/' . $help_file;
		}

		$content = file_get_contents($help_path);

		$html = Markdown::convert($content, false);

		$html = str_replace('code>', 'key>', $html);

		return $html;
	} else {
		$t = Renderer::getMarkupTemplate('settings.tpl', 'addon/advancedcontentfilter/');
		return Renderer::replaceMacros($t, [
			'$messages' => [
				'backtosettings'    => DI::l10n()->t('Back to Addon Settings'),
				'title'             => DI::l10n()->t('Advanced Content Filter'),
				'add_a_rule'        => DI::l10n()->t('Add a Rule'),
				'help'              => DI::l10n()->t('Help'),
				'intro'             => DI::l10n()->t('Add and manage your personal content filter rules in this screen. Rules have a name and an arbitrary expression that will be matched against post data. For a complete reference of the available operations and variables, check the help page.'),
				'your_rules'        => DI::l10n()->t('Your rules'),
				'no_rules'          => DI::l10n()->t('You have no rules yet! Start adding one by clicking on the button above next to the title.'),
				'disabled'          => DI::l10n()->t('Disabled'),
				'enabled'           => DI::l10n()->t('Enabled'),
				'disable_this_rule' => DI::l10n()->t('Disable this rule'),
				'enable_this_rule'  => DI::l10n()->t('Enable this rule'),
				'edit_this_rule'    => DI::l10n()->t('Edit this rule'),
				'edit_the_rule'     => DI::l10n()->t('Edit the rule'),
				'save_this_rule'    => DI::l10n()->t('Save this rule'),
				'delete_this_rule'  => DI::l10n()->t('Delete this rule'),
				'rule'              => DI::l10n()->t('Rule'),
				'close'             => DI::l10n()->t('Close'),
				'addtitle'          => DI::l10n()->t('Add new rule'),
				'rule_name'         => DI::l10n()->t('Rule Name'),
				'rule_expression'   => DI::l10n()->t('Rule Expression'),
				'cancel'            => DI::l10n()->t('Cancel'),
			],
			'$current_theme' => DI::appHelper()->getCurrentTheme(),
			'$rules' => DBA::toArray(DBA::select('advancedcontentfilter_rules', [], ['uid' => DI::userSession()->getLocalUserId()])),
			'$form_security_token' => BaseModule::getFormSecurityToken()
		]);
	}
}

/*
 * Common functions
 */
function advancedcontentfilter_build_fields($data)
{
	$fields = [];

	if (!empty($data['name'])) {
		$fields['name'] = $data['name'];
	}

	if (!empty($data['expression'])) {
		// Using a dummy item to validate the field existence
		$condition = ["(`uid` = ? OR `uid` = 0)", DI::userSession()->getLocalUserId()];
		$params = ['order' => ['uid' => true]];
		$item_row = Post::selectFirstForUser(DI::userSession()->getLocalUserId(), [], $condition, $params);

		if (!DBA::isResult($item_row)) {
			throw new HTTPException\NotFoundException(DI::l10n()->t('This addon requires this node having at least one post'));
		}

		$expressionLanguage = new ExpressionLanguage\ExpressionLanguage();
		$parsedExpression = $expressionLanguage->parse(
			$data['expression'],
			array_keys(advancedcontentfilter_get_filter_fields(advancedcontentfilter_prepare_item_row($item_row)))
		);

		$serialized = serialize($parsedExpression->getNodes());

		$fields['expression'] = $data['expression'];
		$fields['serialized'] = $serialized;
	}

	if (isset($data['active'])) {
		$fields['active'] = intval($data['active']);
	} else {
		$fields['active'] = 1;
	}

	return $fields;
}

/*
 * API
 */

function advancedcontentfilter_get_rules(ServerRequestInterface $request, ResponseInterface $response): ResponseInterface
{
	if (!DI::userSession()->getLocalUserId()) {
		throw new HTTPException\UnauthorizedException(DI::l10n()->t('You must be logged in to use this method'));
	}

	$rules = DBA::toArray(DBA::select('advancedcontentfilter_rules', [], ['uid' => DI::userSession()->getLocalUserId()]));

	$response->getBody()->write(json_encode($rules));
	return $response->withHeader('Content-Type', 'application/json');
}

function advancedcontentfilter_get_rules_id(ServerRequestInterface $request, ResponseInterface $response, $args)
{
	if (!DI::userSession()->getLocalUserId()) {
		throw new HTTPException\UnauthorizedException(DI::l10n()->t('You must be logged in to use this method'));
	}

	$rule = DBA::selectFirst('advancedcontentfilter_rules', [], ['id' => $args['id'], 'uid' => DI::userSession()->getLocalUserId()]);

	$response->getBody()->write(json_encode($rule));
	return $response->withHeader('Content-Type', 'application/json');
}

function advancedcontentfilter_post_rules(ServerRequestInterface $request, ResponseInterface $response)
{
	if (!DI::userSession()->getLocalUserId()) {
		throw new HTTPException\UnauthorizedException(DI::l10n()->t('You must be logged in to use this method'));
	}

	if (!BaseModule::checkFormSecurityToken()) {
		throw new HTTPException\BadRequestException(DI::l10n()->t('Invalid form security token, please refresh the page.'));
	}

	$data = json_decode($request->getBody(), true);

	try {
		$fields = advancedcontentfilter_build_fields($data);
	} catch (Exception $e) {
		throw new HTTPException\BadRequestException($e->getMessage(), $e);
	}

	if (empty($fields['name']) || empty($fields['expression'])) {
		throw new HTTPException\BadRequestException(DI::l10n()->t('The rule name and expression are required.'));
	}

	$fields['uid'] = DI::userSession()->getLocalUserId();
	$fields['created'] = DateTimeFormat::utcNow();

	if (!DBA::insert('advancedcontentfilter_rules', $fields)) {
		throw new HTTPException\ServiceUnavailableException(DBA::errorMessage());
	}

	$rule = DBA::selectFirst('advancedcontentfilter_rules', [], ['id' => DBA::lastInsertId()]);

	DI::cache()->delete('rules_' . DI::userSession()->getLocalUserId());

	$response->getBody()->write(json_encode(['message' => DI::l10n()->t('Rule successfully added'), 'rule' => $rule]));
	return $response->withHeader('Content-Type', 'application/json');
}

function advancedcontentfilter_put_rules_id(ServerRequestInterface $request, ResponseInterface $response, $args)
{
	if (!DI::userSession()->getLocalUserId()) {
		throw new HTTPException\UnauthorizedException(DI::l10n()->t('You must be logged in to use this method'));
	}

	if (!BaseModule::checkFormSecurityToken()) {
		throw new HTTPException\BadRequestException(DI::l10n()->t('Invalid form security token, please refresh the page.'));
	}

	if (!DBA::exists('advancedcontentfilter_rules', ['id' => $args['id'], 'uid' => DI::userSession()->getLocalUserId()])) {
		throw new HTTPException\NotFoundException(DI::l10n()->t('Rule doesn\'t exist or doesn\'t belong to you.'));
	}

	$data = json_decode($request->getBody(), true);

	try {
		$fields = advancedcontentfilter_build_fields($data);
	} catch (Exception $e) {
		throw new HTTPException\BadRequestException($e->getMessage(), $e);
	}

	if (!DBA::update('advancedcontentfilter_rules', $fields, ['id' => $args['id']])) {
		throw new HTTPException\ServiceUnavailableException(DBA::errorMessage());
	}

	DI::cache()->delete('rules_' . DI::userSession()->getLocalUserId());

	$response->getBody()->write(json_encode(['message' => DI::l10n()->t('Rule successfully updated')]));
	return $response->withHeader('Content-Type', 'application/json');
}

function advancedcontentfilter_delete_rules_id(ServerRequestInterface $request, ResponseInterface $response, $args)
{
	if (!DI::userSession()->getLocalUserId()) {
		throw new HTTPException\UnauthorizedException(DI::l10n()->t('You must be logged in to use this method'));
	}

	if (!BaseModule::checkFormSecurityToken()) {
		throw new HTTPException\BadRequestException(DI::l10n()->t('Invalid form security token, please refresh the page.'));
	}

	if (!DBA::exists('advancedcontentfilter_rules', ['id' => $args['id'], 'uid' => DI::userSession()->getLocalUserId()])) {
		throw new HTTPException\NotFoundException(DI::l10n()->t('Rule doesn\'t exist or doesn\'t belong to you.'));
	}

	if (!DBA::delete('advancedcontentfilter_rules', ['id' => $args['id']])) {
		throw new HTTPException\ServiceUnavailableException(DBA::errorMessage());
	}

	DI::cache()->delete('rules_' . DI::userSession()->getLocalUserId());

	$response->getBody()->write(json_encode(['message' => DI::l10n()->t('Rule successfully deleted')]));
	return $response->withHeader('Content-Type', 'application/json');
}

function advancedcontentfilter_get_variables_guid(ServerRequestInterface $request, ResponseInterface $response, $args)
{
	if (!DI::userSession()->getLocalUserId()) {
		throw new HTTPException\UnauthorizedException(DI::l10n()->t('You must be logged in to use this method'));
	}

	if (!isset($args['guid'])) {
		throw new HTTPException\BadRequestException(DI::l10n()->t('Missing argument: guid.'));
	}

	$condition = ["`guid` = ? AND (`uid` = ? OR `uid` = 0)", $args['guid'], DI::userSession()->getLocalUserId()];
	$params = ['order' => ['uid' => true]];
	$item_row = Post::selectFirstForUser(DI::userSession()->getLocalUserId(), [], $condition, $params);

	if (!DBA::isResult($item_row)) {
		throw new HTTPException\NotFoundException(DI::l10n()->t('Unknown post with guid: %s', $args['guid']));
	}

	$return = advancedcontentfilter_get_filter_fields(advancedcontentfilter_prepare_item_row($item_row));

	$response->getBody()->write(json_encode(['variables' => str_replace('\\\'', '\'', var_export($return, true))]));
	return $response->withHeader('Content-Type', 'application/json');
}

/**
 * This mimimcs the processing performed in Model\Item::prepareBody
 *
 * @param array $item_row
 * @return array
 * @throws HTTPException\InternalServerErrorException
 * @throws ImagickException
 */
function advancedcontentfilter_prepare_item_row(array $item_row): array
{
	$tags = Tag::populateFromItem($item_row);

	$item_row['tags'] = $tags['tags'];
	$item_row['hashtags'] = $tags['hashtags'];
	$item_row['mentions'] = $tags['mentions'];
	$item_row['attachments'] = DI::postMediaRepository()->splitAttachments($item_row['uri-id']);

	return $item_row;
}
