<?php

// Copyright (C) 2010-2024, the Friendica project
// SPDX-FileCopyrightText: 2010-2024 the Friendica project
//
// SPDX-License-Identifier: AGPL-3.0-or-later

namespace Friendica\Module\Contact;

use Friendica\App;
use Friendica\BaseModule;
use Friendica\Content\Widget;
use Friendica\Core\Config\Capability\IManageConfigValues;
use Friendica\Core\L10n;
use Friendica\Core\PConfig\Capability\IManagePersonalConfigValues;
use Friendica\Core\Renderer;
use Friendica\Core\Search;
use Friendica\Core\Session\Capability\IHandleUserSessions;
use Friendica\Database\Database;
use Friendica\Model\Contact;
use Friendica\Model\Profile;
use Friendica\Module\Contact as ModuleContact;
use Friendica\Module\Response;
use Friendica\Navigation\SystemMessages;
use Friendica\Network\HTTPClient\Capability\ICanSendHttpRequests;
use Friendica\Network\HTTPClient\Client\HttpClientRequest;
use Friendica\Network\HTTPException\InternalServerErrorException;
use Friendica\Util\Profiler;
use Psr\Log\LoggerInterface;

/**
 * It takes keywords from your profile and queries the directory server for
 * matching keywords from other profiles.
 */
class MatchInterests extends BaseModule
{
	const FETCH_PER_PAGE = 100;

	/** @var IHandleUserSessions */
	protected $session;
	/** @var Database */
	protected $database;
	/** @var SystemMessages */
	protected $systemMessages;
	/** @var App\Page */
	protected $page;
	/** @var App\Mode */
	protected $mode;
	/** @var IManageConfigValues */
	protected $config;
	/** @var IManagePersonalConfigValues */
	protected $pConfig;
	/** @var ICanSendHttpRequests */
	protected $httpClient;

	public function __construct(L10n $l10n, App\BaseURL $baseUrl, App\Arguments $args, LoggerInterface $logger, Profiler $profiler, Response $response, IHandleUserSessions $session, Database $database, SystemMessages $systemMessages, App\Page $page, App\Mode $mode, IManageConfigValues $config, IManagePersonalConfigValues $pConfig, ICanSendHttpRequests $httpClient, array $server, array $parameters = [])
	{
		parent::__construct($l10n, $baseUrl, $args, $logger, $profiler, $response, $server, $parameters);

		$this->session        = $session;
		$this->database       = $database;
		$this->systemMessages = $systemMessages;
		$this->page           = $page;
		$this->mode           = $mode;
		$this->config         = $config;
		$this->pConfig        = $pConfig;
		$this->httpClient     = $httpClient;
	}

	protected function content(array $request = []): string
	{
		if (!$this->session->getLocalUserId()) {
			$this->systemMessages->addNotice($this->t('Permission denied.'));
			$this->baseUrl->redirect('login&return_path=match');
		}

		$profile = Profile::getByUID($this->session->getLocalUserId());

		if (empty($profile)) {
			$this->logger->warning('Couldn\'t find Profile for user id in session.', ['uid' => $this->session->getLocalUserId()]);
			throw new InternalServerErrorException($this->t('Invalid request.'));
		}

		$this->page['aside'] .= Widget::findPeople();
		$this->page['aside'] .= Widget::follow();

		if (empty($profile['pub_keywords']) && empty($profile['prv_keywords'])) {
			$o = Renderer::replaceMacros(Renderer::getMarkupTemplate('contact/list.tpl'), [
				'$title'           => $this->t('Profile Match'),
				'$additional_text' => $this->t('No keywords to match. Please add keywords to your profile.')
			]);
			return $o;
		}

		if ($this->mode->isMobile()) {
			$limit = $this->pConfig->get($this->session->getLocalUserId(), 'system', 'itemspage_mobile_network')
					 ?? $this->config->get('system', 'itemspage_network_mobile');
		} else {
			$limit = $this->pConfig->get($this->session->getLocalUserId(), 'system', 'itemspage_network')
					 ?? $this->config->get('system', 'itemspage_network');
		}

		$searchParameters = [
			's' => trim($profile['pub_keywords'] . ' ' . $profile['prv_keywords']),
			'n' => self::FETCH_PER_PAGE,
		];

		$entries = [];

		foreach ([Search::getGlobalDirectory(), $this->baseUrl] as $server) {
			if (empty($server)) {
				continue;
			}

			try {
				$result = $this->httpClient->post($server . '/search/user/tags', $searchParameters, [], 0, HttpClientRequest::CONTACTDISCOVER);
			} catch (\Throwable $th) {
				$this->logger->notice('Got exception', ['code' => $th->getCode(), 'message' => $th->getMessage()]);
				continue;
			}
			if (!$result->isSuccess()) {
				// try legacy endpoint
				try {
					$result = $this->httpClient->post($server . '/msearch', $searchParameters, [], 0, HttpClientRequest::CONTACTDISCOVER);
				} catch (\Throwable $th) {
					$this->logger->notice('Got exception', ['code' => $th->getCode(), 'message' => $th->getMessage()]);
					continue;
				}
				if (!$result->isSuccess()) {
					$this->logger->notice('Search-Endpoint not available for server.', ['server' => $server]);
					continue;
				}
			}

			$entries = $this->parseContacts(json_decode($result->getBodyString()), $entries, $limit);
		}

		$additional_text = '';
		if (empty($entries)) {
			$additional_text = $this->t('No matches');
		}

		$tpl = Renderer::getMarkupTemplate('contact/list.tpl');
		return Renderer::replaceMacros($tpl, [
			'$title'           => $this->t('Profile Match'),
			'$contacts'        => array_slice($entries, 0, $limit),
			'$additional_text' => $additional_text,
		]);
	}

	/**
	 * parses the JSON result and adds the new entries until the limit is reached
	 *
	 * @param       $jsonResult
	 * @param array $entries
	 * @param int   $limit
	 *
	 * @return array the new entries array
	 */
	protected function parseContacts($jsonResult, array $entries, int $limit): array
	{
		if (empty($jsonResult->results)) {
			return $entries;
		}

		foreach ($jsonResult->results as $profile) {
			if (!$profile) {
				continue;
			}

			// Already known contact
			$contact = Contact::getByURL($profile->url, null, ['rel'], $this->session->getLocalUserId());
			if (!empty($contact) && in_array($contact['rel'], [Contact::FRIEND, Contact::SHARING])) {
				continue;
			}

			$contact = Contact::getByURLForUser($profile->url, $this->session->getLocalUserId());
			if (!empty($contact)) {
				$entries[$contact['id']] = ModuleContact::getContactTemplateVars($contact);
			}

			if (count($entries) == $limit) {
				break;
			}
		}
		return $entries;
	}
}
